
// CONFIG1
#pragma config FEXTOSC = OFF    // FEXTOSC External Oscillator mode Selection bits (Oscillator not enabled)
#pragma config RSTOSC = HFINT1  // Power-up default value for COSC bits (HFINTOSC (1MHz))
#pragma config CLKOUTEN = OFF   // Clock Out Enable bit (CLKOUT function is disabled; I/O or oscillator function on OSC2)
#pragma config CSWEN = ON       // Clock Switch Enable bit (Writing to NOSC and NDIV is allowed)
#pragma config FCMEN = ON       // Fail-Safe Clock Monitor Enable (Fail-Safe Clock Monitor is enabled)

// CONFIG2
#pragma config MCLRE = OFF      // Master Clear Enable bit (MCLR/VPP pin function is digital input; MCLR internally disabled; Weak pull-up under control of port pin's WPU control bit.)
#pragma config PWRTE = OFF      // Power-up Timer Enable bit (PWRT disabled)
#pragma config WDTE = SLEEP     // Watchdog Timer Enable bits (WDT enabled while running and disabled in SLEEP/IDLE; SWDTEN is ignored)
#pragma config LPBOREN = ON     // Low-power BOR enable bit (ULPBOR enabled)
#pragma config BOREN = OFF      // Brown-out Reset Enable bits (Brown-out Reset disabled)
#pragma config BORV = HIGH      // Brown-out Reset Voltage selection bit (Brown-out voltage (Vbor) set to 2.7V)
#pragma config PPS1WAY = ON     // PPSLOCK bit One-Way Set Enable bit (The PPSLOCK bit can be cleared and set only once; PPS registers remain locked after one clear/set cycle)
#pragma config STVREN = ON      // Stack Overflow/Underflow Reset Enable bit (Stack Overflow or Underflow will cause a Reset)
#pragma config DEBUG = OFF      // Debugger enable bit (Background debugger disabled)

// CONFIG3
#pragma config WRT = OFF        // User NVM self-write protection bits (Write protection off)
#pragma config LVP = OFF        // Low Voltage Programming Enable bit (High Voltage on MCLR/VPP must be used for programming.)

// CONFIG4
#pragma config CP = ON          // User NVM Program Memory Code Protection bit (User NVM code protection enabled)
#pragma config CPD = ON         // Data NVM Memory Code Protection bit (Data NVM code protection enabled)

#include <xc.h>

#include "i2c.h"

#define _XTAL_FREQ      4000000

#define LED1RED         LATAbits.LATA5
#define LED1GREEN       LATCbits.LATC1

#define LED2RED         LATAbits.LATA4
#define LED2GREEN       LATCbits.LATC5

#define LED3RED         LATAbits.LATA2
#define LED3GREEN       LATCbits.LATC0

#define ADCVOLTS        LATAbits.LATA0

#define PGC             LATAbits.LATA1

#define TSL_VDD         LATCbits.LATC6

#define BUTTON_UP       PORTCbits.RC4
#define BUTTON_DOWN     PORTBbits.RB7
#define BUTTON_LEFT     PORTAbits.RA3
#define BUTTON_RIGHT    PORTCbits.RC7
#define BUTTON_PUSH     PORTCbits.RC3

#define INTERRUPT_PIN   PORTBbits.RB4

#define INT_PULLUP      WPUBbits.WPUB4
#define SCL_PULLUP      WPUBbits.WPUB5
#define SDA_PULLUP      WPUBbits.WPUB6

#define TSL_ADDR_W     0x52
#define TSL_ADDR_R     0x53

#define COMMAND         0b10100000

#define ENABLE_REG      0x00
#define PERSIST_REG     0x0C
#define CONFIG_REG      0x01
#define AIHTL_REG       0X06
#define C0DATAL_REG     0x14

#define TIMER0_PRELOAD_600S_L    0xC9
#define TIMER0_PRELOAD_600S_H    0xCA

#define TIMER0_PRELOAD_30S_L    0x39
#define TIMER0_PRELOAD_30S_H    0xFE

#define SENSOR_INT          IOCBNbits.IOCBN4
#define SENSOR_INT_FLAG     IOCBFbits.IOCBF4

#define PUSH_INT            IOCCNbits.IOCCN3
#define PUSH_INT_FLAG       IOCCFbits.IOCCF3

#define TIMER_INT           PIE0bits.TMR0IE
#define TIMER_INT_FLAG      PIR0bits.TMR0IF

#define IN_ALARM_STATE      1
#define NOT_ALARM_STATE     0

#define SET_LEVEL_OFFSET    20

#define ENABLE_TIMEOUT      1
#define DISABLE_TIMEOUT     0

#define QUICK_PUSH          0
#define TWO_SEC_PUSH        1
#define TIMEOUT             2

// put the tries counter in eeprom - initialised at program time only
static __eeprom uint8_t tries = 0;
static __eeprom uint16_t alarmcount = 0;

static __eeprom uint8_t code[8] = {1, 1, 1, 1, 1, 1, 1, 1};

// This is the special printout to the screen which
// proves the unit hasn't been tampered with.  It's effectively a unique random
// number for each unit.  Somebody tampering wouldn't know what this is set
// to.
// It is set the first time the unit is armed, based on the light level it
// can see.  It is effectively a random number that lives with the life of the unit.
static __eeprom uint16_t safetynumber = 0;

typedef enum
{
    OFF,
    PIN_CHANGE,
    STANDBY,
    ARMED,
    ENTER_PIN,
    ALARM
} state_t;

// this puts the state as OFF into the EEPROM at program time only
// intentional because we want the state to persist across power removal
static __eeprom state_t state = OFF;

static void tsl_setup(void)
{ 
    i2c_start();
    i2c_write(TSL_ADDR_W);
    i2c_write(COMMAND | PERSIST_REG);
    // trigger always to start
    i2c_write(0x00);
    i2c_stop(); 
    
    i2c_start();
    i2c_write(TSL_ADDR_W);
    i2c_write(COMMAND | CONFIG_REG);
    // start with high gain, max integration time
    i2c_write(0b00100101);
    i2c_stop();  
    
    i2c_start();
    i2c_write(TSL_ADDR_W);
    i2c_write(COMMAND | ENABLE_REG);
    i2c_write(0b00010011);
    i2c_stop();
}

static void low_gain(void)
{
    i2c_start();
    i2c_write(TSL_ADDR_W);
    i2c_write(COMMAND | 0x01);
    // low gain
    i2c_write(0b00000101);
    i2c_stop(); 
}

static void high_gain(void)
{
    i2c_start();
    i2c_write(TSL_ADDR_W);
    i2c_write(COMMAND | CONFIG_REG);
    // high gain, max integration time
    i2c_write(0b00100101);
    i2c_stop(); 
}

static void int_threshold(void)
{
    i2c_start();
    i2c_write(TSL_ADDR_W);
    i2c_write(COMMAND | PERSIST_REG);
    // trigger when over threshold
    i2c_write(0x01);
    i2c_stop(); 
}

static void tsl_off(void)
{
    i2c_start();
    i2c_write(TSL_ADDR_W);
    i2c_write(COMMAND | ENABLE_REG);
    i2c_write(0b00010010);
    i2c_stop();
    
    SENSOR_INT_FLAG = 0;
}

static void clear_int(void)
{
    i2c_start();
    i2c_write(TSL_ADDR_W);
    i2c_write(0b11100110);
    i2c_stop();
}

static void set_threshold(uint16_t threshold)
{
    i2c_start();
    i2c_write(TSL_ADDR_W);
    i2c_write(COMMAND | AIHTL_REG);
    i2c_write((uint8_t) threshold);
    i2c_write((uint8_t) (threshold >> 8));
    i2c_stop();
}

static uint16_t read_light(void)
{
    uint16_t adcvalue;

    i2c_start();
    i2c_write(TSL_ADDR_W);
    i2c_write(COMMAND | C0DATAL_REG);
    i2c_start();
    i2c_write(TSL_ADDR_R);
    adcvalue = i2c_read(0);
    adcvalue = (uint16_t) (i2c_read(1) << 8) | adcvalue;
    i2c_stop();

    return adcvalue;
}

uint8_t wait_for_push(uint8_t timeout_enabled)
{
    // return 2 for timeout (if enabled), 1 for a 2 second push, 0 for single push
    uint8_t i;

    // wait for button to be released
    while (!BUTTON_PUSH)
    {
        CLRWDT();
    }

    __delaywdt_ms(50);

    if (timeout_enabled)
    {
        // turn on the 30 second timeout timer
        TIMER_INT = 1;

        T0CON0bits.T0EN = 1;

        TMR0H = TIMER0_PRELOAD_30S_H;
        TMR0L = TIMER0_PRELOAD_30S_L; 
    }

    TIMER_INT_FLAG = 0;
    PUSH_INT_FLAG = 0;
    
    //must be high when sleeping otherwise wont trigger
    while (!BUTTON_PUSH)
    {
        CLRWDT();
    }
    
    // sleep until button pressed
    SLEEP();
    NOP();
    
    // if the timer woke us up
    if (timeout_enabled && TIMER_INT_FLAG == 1)
    {
        TIMER_INT = 0;
        T0CON0bits.T0EN = 0;
        return TIMEOUT;
    }
    
    // check and see if the button is being held for 2 seconds
    for (i = 0; i < 200; i++)
    {
        if (BUTTON_PUSH == 1)
        {
            break;
        }
        __delaywdt_ms(10);
    }

    // turn the timer off before leaving
    T0CON0bits.T0EN = 0;
    
    if (i == 200)
    {
        return TWO_SEC_PUSH;
    }

    return QUICK_PUSH;
}

static void volts(void)
{
    uint16_t result, i;
    uint8_t level;

    ADCVOLTS = 1;

    // turn it on, C2
    ADCON0 = 0b01001001;
    ADCON1 = 0b11000011;

    // FVR, 2.048V
    FVRCON = 0b10000010;

    // wait for a stable state
    __delaywdt_ms(10);

    ADCON0bits.ADGO = 1;
    while (ADCON0bits.ADGO);

    ADCVOLTS = 0;

    // Turn off ADC and FVR
    ADCON0bits.ADON = 0;
    FVRCONbits.FVREN = 0;

    result = ADRESH;
    result = (result << 8) | ADRESL;

    // The ADC is set up on the PCB such that it's 4 mV per count
    // Minimum voltage for the TSL is 2.7V, so reflect that here
    // BOR is also set to 2.7V
    
    if (result > 725) // 2.900V
    {
        level = 6;
    }  
    else if (result > 712) // 2.848V
    {
        level = 5;
    }  
    else if (result > 700) // 2.800V
    {
        level = 4;
    }   
      else if (result > 687) // 2.748V
    {
        level = 3;
    }              
    else if (result > 675) // 2.700V
    {
        level = 2;
    }
    else
    {
        level = 1;
    }

    // We don't want to put all LEDs on at once 'cos we'll knacker the battery, so
    // cycle through them to light them up
    for (i = 0; i < 500; i++)
    {
        if (level > 0) LED1GREEN = 1;
        __delaywdt_ms(1);
        LED1GREEN = 0;

        if (level > 1) LED1RED = 1;
        __delaywdt_ms(1);
        LED1RED = 0;

        if (level > 2) LED2GREEN = 1;
        __delaywdt_ms(1);
        LED2GREEN = 0;

        if (level > 3) LED2RED = 1;
        __delaywdt_ms(1);
        LED2RED = 0;

        if (level > 4) LED3GREEN = 1;
        __delaywdt_ms(1);
        LED3GREEN = 0;

        if (level > 5) LED3RED = 1;
        __delaywdt_ms(1);
        LED3RED = 0;
    }
}

void do_off(void)
{
    uint8_t i;

    LED1RED = 0;
    LED1GREEN = 0;
    LED2RED = 0;
    LED2GREEN = 0;
    LED3RED = 0;
    LED3GREEN = 0;
    ADCVOLTS = 0;
    
    // disable I2C and INT pullups
    INT_PULLUP = 0;
    SCL_PULLUP = 0;
    SDA_PULLUP = 0;
    
    // turn off power to the TSL
    TSL_VDD = 0;
    
    SENSOR_INT = 0;
    PUSH_INT = 1;

    while (1)
    {
        // if it's a 2 second push
        if (wait_for_push(DISABLE_TIMEOUT) == TWO_SEC_PUSH)
        {
            volts();
            LED1GREEN = 1;
            state = STANDBY;
            
            // turn on power to the the TSL
            TSL_VDD = 1;
            
            INT_PULLUP = 1;
            SCL_PULLUP = 1;
            SDA_PULLUP = 1;
            
            return;
        }
    }
}

void do_standby(void)
{
    uint8_t pushresult;
    
    LED1GREEN = 1;
    LED1RED = 0;
    LED2RED = 0;
    LED2GREEN = 0;
    LED3RED = 0;
    LED3GREEN = 0;
    ADCVOLTS = 0;

    SENSOR_INT = 0;
    PUSH_INT = 1;

    pushresult = wait_for_push(ENABLE_TIMEOUT);
    
    // we can go into either off or armed from here
    if (pushresult == QUICK_PUSH)
    {
        state = ARMED;
    }
    else
    {
        // pushresult was either 1 or 2: 2 second hold or timeout.
        // either way, turn off.
        state = OFF;
    }
}

void do_armed(void)
{
    uint16_t i;
    uint16_t adcvalue;

    LED1RED = 1;
    LED1GREEN = 1;
    LED2RED = 0;
    LED2GREEN = 0;
    LED3RED = 0;
    LED3GREEN = 0;
    ADCVOLTS = 0;

    SENSOR_INT = 1;
    PUSH_INT = 0;

    // reset the 15 min counter
    alarmcount = 0;

    tsl_setup();

    // set up a safety value if we haven't done this before
    // it is stored in the eeprom
    if (safetynumber == 0)
    {
        low_gain();
        clear_int();
    
        // do 2 to ensure one is valid
        for (i=0; i<2; i++)
        {
            // wait for a conversaion
            while (INTERRUPT_PIN)
            {
                CLRWDT();
            }
        }
        
        safetynumber = read_light();
        
        high_gain();
    }
    
    // wait for 20 seconds, go back to idle if button pressed
    for (i = 0; i < 20000; i++)
    {
        if (!BUTTON_PUSH)
        {
            state = STANDBY;
            
            // turn the tsl back off - that's what it should be in standby
            tsl_off();
            return;
        }
        __delaywdt_ms(1);
    }

    // read 5 times first so we're stable
    // the interrupt at this point fires for every reading
    for (i = 0; i < 5; i++)
    {
        // wait for a reading
        clear_int();
        while (INTERRUPT_PIN)
        {
            CLRWDT();
        }
    }

    int_threshold();
    
    adcvalue = read_light();

    // deal with if the light is maxed out - we never want to set the level to
    // any more than the max minus 1, so we can always get out by shining a torch on it
    if (adcvalue > (0xFFFF - SET_LEVEL_OFFSET))
    {
        adcvalue = 0xFFFF - SET_LEVEL_OFFSET - 1;
    }

    set_threshold((uint16_t) (adcvalue + SET_LEVEL_OFFSET));

    // armed now
    LED1GREEN = 0;
    
    // set the state to ALARM so that if the power is removed, it goes straight
    // to alarm.  This will be overridden below by ENTER_PIN when some light 
    // is detected.
    state = ALARM;

    __delaywdt_ms(2000);

    LED1RED = 0;

    PUSH_INT_FLAG = 0;
    
    while (1)
    {
        CLRWDT();
        
        clear_int();
        SENSOR_INT_FLAG = 0;
        while (!INTERRUPT_PIN);

        SLEEP();
        NOP();

        adcvalue = read_light();

        if (adcvalue >= 20)
        {
            tsl_off();
            state = ENTER_PIN;
            LED1RED = 1;
            break;
        }
    }
}

void do_enter_pin(uint8_t alarm_state)
{
    uint16_t i;
    uint8_t codecount = 0, codein[8], j, coderight, trycount;

    if (!alarm_state)
    {
        LED1RED = 1;
        LED1GREEN = 0;
        LED2RED = 0;
        LED2GREEN = 1;
        LED3RED = 0;
        LED3GREEN = 0;
        trycount = 3;
    }
    else
    {
        // We came here from the alarm state
        LED1RED = 0;
        LED1GREEN = 0;
        LED2RED = 0;
        LED2GREEN = 1;
        LED3RED = 0;
        LED3GREEN = 0;

        // Only do one PIN try - we are already in alarm state
        trycount = 1;
    }

    ADCVOLTS = 0;

    SENSOR_INT = 0;
    PUSH_INT = 0;

    // number of tries is kept in the eeprom
    while (tries < trycount)
    {
        for (i = 0; i < 30000; i++)
        {
            if (!BUTTON_UP)
            {
                codein[codecount] = 1;
                codecount++;
                while (!BUTTON_UP);
                __delaywdt_ms(50);
            }
            if (!BUTTON_RIGHT)
            {
                codein[codecount] = 2;
                codecount++;
                while (!BUTTON_RIGHT);
                __delaywdt_ms(50);
            }
            if (!BUTTON_DOWN)
            {
                codein[codecount] = 3;
                codecount++;
                while (!BUTTON_DOWN);
                __delaywdt_ms(50);
            }
            if (!BUTTON_LEFT)
            {
                codein[codecount] = 4;
                codecount++;
                while (!BUTTON_LEFT);
                __delaywdt_ms(50);
            }

            if (codecount >= 8)
            {
                // put a 100ms delay here to make brute forcing slow
                __delaywdt_ms(100);
                
                coderight = 1; 
                for (j = 0; j < 8; j++)
                {
                    if (code[j] != codein[j])
                    {
                        coderight = 0;
                    }
                }

                break;
            }
            __delaywdt_ms(1);
        }

        if (i == 30000)
        {
            // we timed out, so go straight to alarm
            tries = 3;
        }
        else if (coderight == 1)
        {
            // if the next state is STANDBY then show the safety number
            // here, to prove that nothing has been tampered with
            LED1GREEN = (safetynumber & 0x01) ? 1 : 0;
            LED1RED = (safetynumber & 0x02) ? 1 : 0;
            LED2GREEN = (safetynumber & 0x04) ? 1 : 0;
            LED2RED = (safetynumber & 0x08) ? 1 : 0;
            LED3GREEN = (safetynumber & 0x10) ? 1 : 0;
            LED3RED = (safetynumber & 0x20) ? 1 : 0;

            __delaywdt_ms(5000);
            
            state = STANDBY;
            tries = 0;
            return;
        }
        else
        {
            // bad pin
            tries++;
            LED2RED = 1;
            __delaywdt_ms(2000);
            LED2RED = 0;
            codecount = 0;
        }
    }

    // number of tries exhausted
    state = ALARM;
    tries = 0;
}

void do_alarm(void)
{
    uint16_t i, count;

    LED1RED = 0;
    LED1GREEN = 0;
    LED2RED = 1;
    LED2GREEN = 0;
    LED3RED = 0;
    LED3GREEN = 0;
    ADCVOLTS = 0;

    TIMER_INT = 1;
    PUSH_INT = 1;
    SENSOR_INT = 0;

    PUSH_INT_FLAG = 0;
    SENSOR_INT_FLAG = 0;

    T0CON0bits.T0EN = 1;

    TMR0H = TIMER0_PRELOAD_600S_H;
    TMR0L = TIMER0_PRELOAD_600S_L;

    TIMER_INT_FLAG = 0;
    PUSH_INT_FLAG = 0;

    // The timer rolls over every 15 minutes - increment counter when it does
    while (1)
    {
        SLEEP();
        NOP();

        if (TIMER_INT_FLAG == 1)
        {
            alarmcount++;
            TMR0H = TIMER0_PRELOAD_600S_H;
            TMR0L = TIMER0_PRELOAD_600S_L;
            TIMER_INT_FLAG = 0;
        }

        if (PUSH_INT_FLAG == 1)
        {
            count = 0;

            // If it's a short push then show the elapsed alarm time, otherwise go to entering pin
            while (!BUTTON_PUSH)
            {
                if (count > 2000)
                {
                    break;
                }
                count++;
                __delaywdt_ms(1);
            }

            // Short press
            if (count < 2000)
            {
                // Pause the timer
                T0CON0bits.T0EN = 0;
                for (i = 0; i < alarmcount; i++)
                {
                    LED3RED = 1;
                    __delaywdt_ms(100);
                    LED3RED = 0;
                    __delaywdt_ms(500);
                }

                // Resume the timer
                T0CON0bits.T0EN = 1;
            }
            else
            {
                T0CON0bits.T0EN = 0;
                do_enter_pin(IN_ALARM_STATE);
                return;
            }
            PUSH_INT_FLAG = 0;
        }
    }
}

void do_pin_change(void)
{
    uint16_t i;
    uint8_t codein[16], codecount = 0, correct;

    LED1RED = 0;
    LED1GREEN = 1;
    LED2RED = 1;
    LED2GREEN = 1;
    LED3RED = 0;
    LED3GREEN = 0;
    ADCVOLTS = 0;

    SENSOR_INT = 0;
    PUSH_INT = 1;

    // Read the code in twice within 20 seconds. First 8 has to match last 8 (to check you typed it right)
    for (i = 0; i < 20000; i++)
    {
        if (!BUTTON_UP)
        {
            codein[codecount] = 1;
            codecount++;
            while (!BUTTON_UP);
            __delaywdt_ms(50);
        }
        if (!BUTTON_RIGHT)
        {
            codein[codecount] = 2;
            codecount++;
            while (!BUTTON_RIGHT);
            __delaywdt_ms(50);
        }
        if (!BUTTON_DOWN)
        {
            codein[codecount] = 3;
            codecount++;
            while (!BUTTON_DOWN);
            __delaywdt_ms(50);
        }
        if (!BUTTON_LEFT)
        {
            codein[codecount] = 4;
            codecount++;
            while (!BUTTON_LEFT);
            __delaywdt_ms(50);
        }

        if (codecount == 8)
        {
            // Put the LED on after the first 8 to tell the user they are now retyping the new pin
            LED3GREEN = 1;
        }

        if (codecount >= 16)
        {
            break;
        }

        __delaywdt_ms(1);
    }

    // Timeout
    if (i == 20000)
    {
        state = STANDBY;
        return;
    }

    // Now check that the two typed codes match each other
    correct = 1;
    for (i = 0; i < 8; i++)
    {
        if (codein[i] != codein[i + 8])
        {
            correct = 0;
            break;
        }
    }

    if (correct == 1)
    {
        LED2RED = 0;
        LED1GREEN = 0;
        LED2GREEN = 0;
        LED3GREEN = 0;
        __delaywdt_ms(100);
        LED1GREEN = 1;
        LED2GREEN = 1;
        LED3GREEN = 1;
        __delaywdt_ms(100);
        LED1GREEN = 0;
        LED2GREEN = 0;
        LED3GREEN = 0;
        __delaywdt_ms(100);
        LED1GREEN = 1;
        LED2GREEN = 1;
        LED3GREEN = 1;
        __delaywdt_ms(100);

        // Now change the code
        for (i = 0; i < 8; i++)
        {
            code[i] = codein[i];
        }
    }
    else
    {
        LED2RED = 0;
        LED1GREEN = 0;
        LED2GREEN = 0;
        LED3GREEN = 0;
        __delaywdt_ms(100);
        LED1RED = 1;
        LED2RED = 1;
        LED3RED = 1;
        __delaywdt_ms(100);
        LED1RED = 0;
        LED2RED = 0;
        LED3RED = 0;
        __delaywdt_ms(100);
        LED1RED = 1;
        LED2RED = 1;
        LED3RED = 1;
        __delaywdt_ms(100);
    }

    state = STANDBY;
}

void main(void)
{
    uint8_t i;
    uint16_t adcvalue;

    ANSELA = 0;
    ANSELB = 0;
    ANSELC = 0b00000100;

    LATA = 0;
    LATB = 0;
    LATC = 0;

    WPUC = 0b10011000;
    WPUB = 0b11110000;
    WPUA = 0b00001000;
            
    TRISA = 0b00001000;
    TRISB = 0b11110000;
    TRISC = 0b10011100;

    // SCL
    RB5PPS = 0x18;
    SSP1CLKPPS = 0x0D;

    // SDA
    RB6PPS = 0x19;
    SSP1DATPPS = 0x0E;

    // save power
    PGC = 0;
    
    // enable power to the tsl
    TSL_VDD = 1;

    // Disable modules we're not going to use (disable all then selectively re-enable)
    PMD0 = 0xFF;
    PMD1 = 0xFF;
    PMD2 = 0xFF;
    PMD3 = 0xFF;
    PMD4 = 0xFF;
    PMD5 = 0xFF;
    PMD0bits.NVMMD = 0;
    PMD0bits.IOCMD = 0;
    PMD0bits.FVRMD = 0;
    PMD1bits.TMR0MD = 0;
    PMD2bits.ADCMD = 0;
    PMD4bits.MSSP1MD = 0;

    // Set oscillator to 4 MHz
    OSCCON1 = 0b01100000;
    OSCFRQ = 0x03;
    
    // See if the watchdog timed out.  Sit here indefinitely for the time being
    if (STATUSbits.nTO == 0)
    {
        while (1)
        {
            LED3RED = 1;
            __delaywdt_ms(100);
            LED3RED = 0;
            __delaywdt_ms(100);
        }
    }

    // INTERRUPT ENABLE
    PIE0bits.IOCIE = 1;
    INTCONbits.PEIE = 1; //enables peripheral interrupt

    // TIMER bits
    T0CON0bits.T0EN = 0;
    T0CON0bits.T016BIT = 1;
    T0CON1bits.T0CS = 0b100; // LFINTOSC
    T0CON1bits.T0ASYNC = 1;
    T0CON1bits.T0CKPS = 0b1011; // Divide by 2048. Freq is 31kHz.  
    // Preload timer with 51913 means it rolls after 899.9969 secs.
    // Or preload with 65081 to rollover after 29.9932 secs

    i2c_init();
    __delaywdt_ms(10);

    // put the tsl in low power state
    tsl_off();

    // On power up, if state is OFF or STANDBY and button is held then we want to change the PIN
    if ((state == OFF) || (state == STANDBY))
    {
        if (!BUTTON_PUSH)
        {
            // Request the PIN and change it if successful
            do_pin_change();
        }
        else
        {
            // show the voltage when you plug a new coin cell in
            // but only if we're not changing the PIN
            
            // only show the voltage if its OFF or STANDBY
            volts();
        }
    }
    
    while (1)
    {
        CLRWDT();

        switch (state)
        {
            case OFF:
                do_off();
                break;

            case STANDBY:
                do_standby();
                break;

            case ARMED:
                do_armed();
                break;

            case ENTER_PIN:
                do_enter_pin(NOT_ALARM_STATE);
                break;

            case ALARM:
                do_alarm();
                break;

            case PIN_CHANGE:
                do_pin_change();
                break;

            default:
                break;
        }
    }
}
